RM1_ResOps <- function(target_plants_mapped_to_water,
                       resOps_releases){
  
  expand.grid(
    date = seq.Date(ymd(paste0(earliest_EIA_year, "-01-01")),
             ymd("2019-12-31"),
             by = 1),
    RHPID = unique(target_plants_mapped_to_water[["RHPID"]])
  ) |> as_tibble() -> release_table_to_fill

  resOps_releases |> 
    pivot_longer(-date, names_to = "grand_id", values_to = "RM1_ResOps") ->
    resOps_releases_long

  target_plants_mapped_to_water |> 
    select(RHPID, grand_id) |> 
    filter(!is.na(grand_id)) |> 
    mutate(grand_id = as.character(grand_id)) |> 
    unique() ->
    RHPID_to_grand
  
  release_table_to_fill |> 
    left_join(RHPID_to_grand, by = join_by(RHPID)) |> 
    left_join(resOps_releases_long,
              by = join_by(date, grand_id)) |> 
    select(-grand_id) ->
    RM1_ResOps_ready
  
  # ResOps cases to remove
  c(
    "414_BEARDSLEY",     # ResOps provides non-power release
    "416_TULLOCH",       # ResOps provides non-power release
    "450_SPRING CREEK",  # Piped diversion, ResOps provides non-power release
    "537_CAMANCHE",      # ResOps provides non-power release
    "2203_HUNGRY HORSE", # ResOps provides non-power release
    "2707_BLEWETT",      # ResOps provides non-power release
    "3319_JEFFERIES"     # ResOps provides non-power release
  ) -> cases_for_removal

  RM1_ResOps_ready |>
      mutate(RM1_ResOps = if_else(RHPID %in% cases_for_removal, NA_real_, RM1_ResOps)) -> RM1_ResOps_ready_filtered
  return(RM1_ResOps_ready_filtered)  

}

RM2_HILARRI <- function(target_plants_mapped_to_water,
                        USGS_flow_parquet){

  expand.grid(
    date = seq.Date(ymd(paste0(earliest_EIA_year, "-01-01")),
                    ymd(paste0(latest_EIA_year, "-12-31")),
                    by = 1),
    RHPID = unique(target_plants_mapped_to_water[["RHPID"]])
  ) |> as_tibble() -> release_table_to_fill

  read_parquet(USGS_flow_parquet) |> 
    mutate(usgs_gage = paste0("USGS-", usgs_gage)) |> 
    unique() -> flow_data

  release_table_to_fill |> 
    left_join(target_plants_mapped_to_water |> 
                filter(!is.na(usgs_gage)) |> 
                select(RHPID, usgs_gage) |> unique(),
              by = join_by(RHPID)) |> 
    left_join(flow_data, join_by(date, usgs_gage)) |> 
    mutate(RM2_USGS = flow_cfs * 0.028316832) |> 
    select(-flow_cfs, -usgs_gage) ->
    RM2_HILARRI_ready
  
  return(RM2_HILARRI_ready)
  
}

RM3_ISTARF <- function(target_plants_mapped_to_water,
                       ISTARF_dir,
                       ResOps_dir,
                       daily_releases){
  
  read_csv(paste0(ISTARF_dir, "/ISTARF-CONUS.csv")) -> ISTARF
  
  expand.grid(
    date = seq.Date(ymd(paste0(earliest_EIA_year, "-01-01")),
                    ymd("2019-12-31"),
                    by = 1),
    RHPID = unique(target_plants_mapped_to_water[["RHPID"]])
  ) |> as_tibble() -> release_table_to_fill

  target_plants_mapped_to_water |>
    filter(grand_id %in% ISTARF[["GRanD_ID"]],
           COMID %in% names(daily_releases)) |> 
    select(RHPID, grand_id, COMID) |> 
    unique() |>  #.[1:3,] |> 
    pmap_dfr(function(RHPID, grand_id, COMID){
      
      message(paste0("simulating ISTARF release for ", RHPID, "..."))

      daily_releases |> 
        select(date, av_flow_cfs = one_of(as.character(COMID))) |> 
        mutate(i = av_flow_cfs * 0.028317 * 60 * 60 * 24 * 1e-6) |> 
        select(-av_flow_cfs) ->
        inflow_MCM
      
      ResOps_fn <- paste0(ResOps_dir, "/time_series_all/ResOpsUS_",
                           grand_id, ".csv")
      
      if(ResOps_fn %in% list.files(paste0(ResOps_dir, "/time_series_all"))){
        
        vroom(paste0(ResOps_dir, "/time_series_all/ResOpsUS_",
                     grand_id, ".csv"),
              progress = FALSE, show_col_types = FALSE) |>
          mutate(r = outflow * 60 * 60 * 24 * 1e-6) |> 
          select(date, s = storage, r) ->
          dam_obs
      }else{
        inflow_MCM |> select(-i) |>
          mutate(s = NA_real_, r = NA_real_) -> dam_obs
      }

      filter(ISTARF, GRanD_ID == grand_id) ->
        ISTARF_attrs
      
      # upper bound of NOR
      convert_parameters_to_targets(
        parameters = c(ISTARF_attrs[["NORhi_mu"]],
                       ISTARF_attrs[["NORhi_alpha"]],
                       ISTARF_attrs[["NORhi_beta"]],
                       ISTARF_attrs[["NORhi_max"]],
                       ISTARF_attrs[["NORhi_min"]]),
        # give the harmonic a name...
        target_name = "NORhi"
      ) -> NOR_upper_bound
      
      # lower bound of NOR
      convert_parameters_to_targets(
        parameters = c(ISTARF_attrs[["NORlo_mu"]],
                       ISTARF_attrs[["NORlo_alpha"]],
                       ISTARF_attrs[["NORlo_beta"]],
                       ISTARF_attrs[["NORlo_max"]],
                       ISTARF_attrs[["NORlo_min"]]),
        # give the harmonic a name...
        target_name = "NORlo"
      ) -> NOR_lower_bound
      
      
      # release harmonic
      convert_parameters_to_release_harmonic(c(
        ISTARF_attrs[["Release_alpha1"]],
        ISTARF_attrs[["Release_beta1"]],
        ISTARF_attrs[["Release_alpha2"]],
        ISTARF_attrs[["Release_beta2"]])
      ) -> release_harmonic
      
      rr <-
        c(
          ISTARF_attrs[["Release_c"]],
          ISTARF_attrs[["Release_p1"]],
          ISTARF_attrs[["Release_p2"]]
        )
      
      #
      capacity_MCM <- ISTARF_attrs[["GRanD_CAP_MCM"]]
      
      release_table_to_fill |> 
        filter(RHPID == !!RHPID) |> 
        mutate(epiweek = epiweek(date),
               epiweek = if_else(epiweek == 53, 52, epiweek)) |> 
        left_join(NOR_upper_bound, by = join_by(epiweek)) |> 
        left_join(NOR_lower_bound, by = join_by(epiweek)) |> 
        left_join(release_harmonic, by = join_by(epiweek)) |> 
        left_join(dam_obs, by = "date") |> 
        left_join(inflow_MCM, by = "date") ->
        sim_data

      sim_date <- sim_data$date
      i <- sim_data$i
      flood_curve <- sim_data$NORhi
      cons_curve <- sim_data$NORlo
      standarized_release_signal <- sim_data$release_harmonic
      
      # x -> observed data; x_ -> computed variable
      s <- sim_data$s
      # Set initial storage to middle of operating range
      if(is.na(s[1])){
        s[1] <- capacity_MCM *
          ((sim_data[["NORhi"]][1] + sim_data[["NORlo"]][1] / 2)) / 100
      } 
      s_ <- s
      a_ <- rep(NA, length(sim_date))
      r <- sim_data$r
      r_ <- r
      inflow_mean <- mean(i) * 7
      r_max <- (inflow_mean * (1 + ISTARF_attrs$Release_max) / 7)
      r_min <- (inflow_mean * (1 + ISTARF_attrs$Release_min) / 7)
      

      for (t in 1:length(i)){
        #message(t)
        # week-ahead persistence forecast
        i_fcast <- i[t] * 7
        (i_fcast / inflow_mean) - 1 -> i_st
        
        # simulation
        a_[t] <- (100 * (s_[t] / capacity_MCM) - cons_curve[t]) / (flood_curve[t] - cons_curve[t])
        
        if(a_[t] < 0){
          r_[t] <- max((i_fcast - ((capacity_MCM * (cons_curve[t] / 100) - s_[t]))) / 7, r_min)
        }else{
          if(a_[t] > 1){
            r_[t] <- (s_[t] - (capacity_MCM * (flood_curve[t] / 100)) + i_fcast) / 7
          }else{
            
            r_[t] <- (inflow_mean *
                        (1 + (
                          standarized_release_signal[t] + rr[1] + (a_[t] * rr[2]) + (i_st * rr[3])
                        ))) / 7
          }
          
          if(r_[t] < 0) r_[t] <- r_min
          if(r_[t] > r_max) r_[t] <- r_max
          if(r_[t] > s_[t] + i[t]) r_[t] <- s_[t] + i[t]
        }
        
        if(!is.na(r[t])) r_[t] <- r[t]
        
        if((s_[t] - r_[t] + i[t]) > capacity_MCM){
          s_[t + 1] <- capacity_MCM
          r_[t] <- s_[t] + i[t] - capacity_MCM
        }else{
          s_[t + 1] <- s_[t] + i[t] - r_[t]
        }
        
        # correct to observed if non-NA
        if(!is.na(r[t])) r_[t] <- r[t]
        if(!is.na(s[t+1])) s_[t+1] <- s[t+1]
        
      }
      
     return(tibble(
       RHPID = !!RHPID,
       date = sim_data$date,
       s = s[1:length(i)],
       s_ = s_[1:length(i)],
       r = r,
       r_ = r_
     ))
      
    }) -> results

  
  
  release_table_to_fill |> 
    left_join(results |>
                select(RHPID, date, RM3_ISTARF = r_),
              by = c("RHPID", "date")) |> 
    # convert to cumecs
    mutate(RM3_ISTARF = RM3_ISTARF * 1e6 / (24 * 60 * 60)) ->
    RM3_ISTARF_ready
  
  return(RM3_ISTARF_ready)
  
  
  # results |> mutate(r = sim_data$r) |> 
  #   mutate(date = unique(release_table_to_fill$date)) |> 
  #   filter(year(date) %in% 2017:2019) |> 
  #   ggplot(aes(date, r)) + geom_line() +
  #   geom_line(aes(y=r_), linetype = 2, col = "red")

}

RM4_RoR <- function(target_plants_mapped_to_water,
                    daily_releases){


  target_plants_mapped_to_water |> 
    select(RHPID, dominant_mode, COMID) |> 
    filter(grepl("Run-of-river", dominant_mode) |
             RHPID %in% c("3084/7511_MCNARY & MCNARY FISH",
                         "3921_CHIEF JOSEPH")) |>
    # ^^ correction for missing op mode in some Columbia RoR plants
    filter(COMID %in% names(daily_releases)) |> 
    filter(!is.na(COMID)) |> unique() ->
    RoR_plants_with_inflow
  
  RoR_plants_with_inflow |> filter(dominant_mode == "Run-of-river/Upstream Peaking")
  
  
  expand.grid(
    date = seq.Date(ymd(paste0(earliest_EIA_year, "-01-01")),
                    ymd(paste0(2019L, "-12-31")),
                    by = 1),
    RHPID = unique(target_plants_mapped_to_water[["RHPID"]])
  ) |> as_tibble() -> release_table_to_fill

  
  daily_releases |> 
    select(date, one_of(as.character(RoR_plants_with_inflow[["COMID"]]))) |> 
    pivot_longer(-date, names_to = "COMID",
                 values_to = "RM4_RoR") |> 
    mutate(COMID = as.integer(COMID)) -> flows_by_COMID
  
  
  release_table_to_fill |> 
    left_join(RoR_plants_with_inflow, by = "RHPID") |> 
    left_join(flows_by_COMID, by = c("date", "COMID")) |> 
    select(-dominant_mode, -COMID) |> 
    # convert to cumecs
    mutate(RM4_RoR = RM4_RoR * 0.028316832) ->
    RM4_RoR_ready

  return(RM4_RoR_ready)
  
}

RM5_HUC4outlet <- function(target_plants_mapped_to_water,
                           USGS_flow_HUC4outlet_parquet){

  expand.grid(
    date = seq.Date(ymd(paste0(earliest_EIA_year, "-01-01")),
                    ymd(paste0(latest_EIA_year, "-12-31")),
                    by = 1),
    RHPID = unique(target_plants_mapped_to_water[["RHPID"]])
  ) |> as_tibble() -> release_table_to_fill
  
  read_parquet(USGS_flow_HUC4outlet_parquet) |> 
    select(date, HUC4, flow_cfs) |> 
    unique() -> flow_data

  release_table_to_fill |> 
    left_join(target_plants_mapped_to_water |> 
                mutate(HUC4 = substr(huc_12, 1, 4)) |> 
                select(RHPID, HUC4) |> unique(),
              by = join_by(RHPID)) |> 
    left_join(flow_data, join_by(date, HUC4)) |> 
    mutate(RM5_HUC4 = flow_cfs * 0.028316832) |> 
    select(-flow_cfs, -HUC4) ->
    RM5_HUC4outlet_ready
  
  return(RM5_HUC4outlet_ready)
  
}
